<?php
/**
 * Exemplo de teste da API do Berapay
 * 
 * Este arquivo demonstra como usar a API para criar pagamentos PIX
 * e consultar status de transações.
 */

// Configurações da API
$api_key = 'SUA_API_KEY_AQUI'; // Substitua pela sua API Key
$api_url = 'https://api.berapay.com.br/v1/gateway/';

// Função para fazer requisições HTTP
function makeRequest($url, $data = null, $method = 'GET') {
    $ch = curl_init();
    
    if ($method === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($data) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
    } elseif ($method === 'GET' && $data) {
        $url .= '?' . http_build_query($data);
    }
    
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Accept: application/json'
    ]);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    
    curl_close($ch);
    
    if ($error) {
        return ['error' => 'cURL Error: ' . $error];
    }
    
    return [
        'status_code' => $http_code,
        'body' => $response,
        'data' => json_decode($response, true)
    ];
}

// Função para criar pagamento PIX
function createPixPayment($api_key, $api_url, $amount, $client_data) {
    $payment_data = [
        'amount' => $amount,
        'client' => $client_data,
        'api-key' => $api_key,
        'postback' => 'https://seusite.com/webhook/pix' // URL do seu webhook
    ];
    
    $url = $api_url . 'criar-pagamento.php';
    
    echo "🔄 Criando pagamento PIX...\n";
    echo "URL: $url\n";
    echo "Dados: " . json_encode($payment_data, JSON_PRETTY_PRINT) . "\n\n";
    
    $response = makeRequest($url, $payment_data, 'POST');
    
    if (isset($response['error'])) {
        echo "❌ Erro: " . $response['error'] . "\n";
        return false;
    }
    
    echo "📡 Resposta da API:\n";
    echo "Status Code: " . $response['status_code'] . "\n";
    echo "Body: " . $response['body'] . "\n\n";
    
    if ($response['status_code'] === 200 && $response['data']) {
        $data = $response['data'];
        
        if ($data['status'] === 'success') {
            echo "✅ Pagamento PIX criado com sucesso!\n";
            echo "ID da Transação: " . $data['data']['idTransaction'] . "\n";
            echo "Referência Externa: " . $data['data']['externalReference'] . "\n";
            echo "Status: " . $data['data']['status'] . "\n";
            echo "Expira em: " . $data['data']['expiresAt'] . "\n";
            echo "Chave PIX: " . $data['data']['pixKey'] . "\n\n";
            
            return $data['data'];
        } else {
            echo "❌ Erro ao criar pagamento: " . $data['message'] . "\n";
            return false;
        }
    } else {
        echo "❌ Erro na requisição. Status: " . $response['status_code'] . "\n";
        return false;
    }
}

// Função para consultar status do pagamento
function checkPaymentStatus($api_key, $api_url, $transaction_id) {
    $url = $api_url . 'consultar-status.php';
    
    $params = [
        'api-key' => $api_key,
        'idTransaction' => $transaction_id
    ];
    
    echo "🔍 Consultando status do pagamento...\n";
    echo "URL: $url\n";
    echo "Parâmetros: " . json_encode($params, JSON_PRETTY_PRINT) . "\n\n";
    
    $response = makeRequest($url, $params, 'GET');
    
    if (isset($response['error'])) {
        echo "❌ Erro: " . $response['error'] . "\n";
        return false;
    }
    
    echo "📡 Resposta da API:\n";
    echo "Status Code: " . $response['status_code'] . "\n";
    echo "Body: " . $response['body'] . "\n\n";
    
    if ($response['status_code'] === 200 && $response['data']) {
        $data = $response['data'];
        
        if ($data['status'] === 'success') {
            echo "✅ Status consultado com sucesso!\n";
            echo "ID da Transação: " . $data['data']['idTransaction'] . "\n";
            echo "Status: " . $data['data']['status'] . "\n";
            echo "Valor: R$ " . $data['data']['amount'] . "\n";
            echo "Cliente: " . $data['data']['client']['name'] . "\n";
            echo "E-mail: " . $data['data']['client']['email'] . "\n";
            
            if (isset($data['data']['asaas_status'])) {
                echo "Status Asaas: " . $data['data']['asaas_status'] . "\n";
            }
            
            return $data['data'];
        } else {
            echo "❌ Erro ao consultar status: " . $data['message'] . "\n";
            return false;
        }
    } else {
        echo "❌ Erro na requisição. Status: " . $response['status_code'] . "\n";
        return false;
    }
}

// Função para simular webhook
function simulateWebhook($transaction_id, $status, $amount) {
    $webhook_data = [
        'idTransaction' => $transaction_id,
        'externalReference' => 'test_' . time(),
        'amount' => $amount,
        'paymentMethod' => 'PIX',
        'status' => $status,
        'timestamp' => date('c')
    ];
    
    echo "📡 Simulando webhook...\n";
    echo "Dados: " . json_encode($webhook_data, JSON_PRETTY_PRINT) . "\n\n";
    
    // Aqui você enviaria os dados para sua URL de webhook
    // $webhook_url = 'https://seusite.com/wc-api/wc_pix_gateway';
    // $response = makeRequest($webhook_url, $webhook_data, 'POST');
    
    echo "✅ Webhook simulado com sucesso!\n\n";
}

// Função principal de teste
function runTests() {
    global $api_key, $api_url;
    
    echo "🚀 Iniciando testes da API do Berapay\n";
    echo "==========================================\n\n";
    
    // Verificar se a API Key foi configurada
    if ($api_key === 'SUA_API_KEY_AQUI') {
        echo "❌ ERRO: Configure sua API Key no arquivo antes de executar os testes!\n";
        echo "Edite o arquivo e substitua 'SUA_API_KEY_AQUI' pela sua chave real.\n\n";
        return;
    }
    
    // Dados do cliente para teste
    $client_data = [
        'name' => 'João Silva Teste',
        'document' => '12345678901',
        'email' => 'joao.teste@email.com',
        'telefone' => '11999999999'
    ];
    
    $amount = 50.00; // Valor de teste
    
    echo "📋 Dados do teste:\n";
    echo "API Key: " . substr($api_key, 0, 10) . "...\n";
    echo "API URL: $api_url\n";
    echo "Cliente: " . $client_data['name'] . "\n";
    echo "Valor: R$ " . number_format($amount, 2, ',', '.') . "\n\n";
    
    // Teste 1: Criar pagamento PIX
    echo "🧪 TESTE 1: Criar Pagamento PIX\n";
    echo "--------------------------------\n";
    
    $payment = createPixPayment($api_key, $api_url, $amount, $client_data);
    
    if (!$payment) {
        echo "❌ Teste 1 falhou. Parando execução.\n";
        return;
    }
    
    echo "✅ Teste 1 concluído com sucesso!\n\n";
    
    // Aguardar um pouco antes do próximo teste
    echo "⏳ Aguardando 3 segundos antes do próximo teste...\n";
    sleep(3);
    
    // Teste 2: Consultar status do pagamento
    echo "🧪 TESTE 2: Consultar Status do Pagamento\n";
    echo "-----------------------------------------\n";
    
    $status = checkPaymentStatus($api_key, $api_url, $payment['idTransaction']);
    
    if (!$status) {
        echo "❌ Teste 2 falhou.\n";
    } else {
        echo "✅ Teste 2 concluído com sucesso!\n";
    }
    
    echo "\n";
    
    // Teste 3: Simular webhook
    echo "🧪 TESTE 3: Simular Webhook\n";
    echo "----------------------------\n";
    
    simulateWebhook($payment['idTransaction'], 'PAID_OUT', $amount);
    
    echo "✅ Teste 3 concluído com sucesso!\n\n";
    
    // Resumo dos testes
    echo "📊 RESUMO DOS TESTES\n";
    echo "====================\n";
    echo "✅ Teste 1 (Criar Pagamento): Concluído\n";
    echo "✅ Teste 2 (Consultar Status): " . ($status ? 'Concluído' : 'Falhou') . "\n";
    echo "✅ Teste 3 (Simular Webhook): Concluído\n\n";
    
    echo "🎉 Todos os testes foram executados!\n";
    echo "Verifique os resultados acima para confirmar que a API está funcionando corretamente.\n\n";
    
    // Informações adicionais
    echo "💡 DICAS:\n";
    echo "- Use o modo sandbox para testes sem valores reais\n";
    echo "- Verifique os logs do WooCommerce para debugging\n";
    echo "- Teste o webhook em um ambiente real\n";
    echo "- Monitore as transações no painel administrativo\n\n";
}

// Executar testes se o arquivo for chamado diretamente
if (php_sapi_name() === 'cli' || isset($_GET['run_tests'])) {
    runTests();
} else {
    echo "<h1>Teste da API do Berapay</h1>";
    echo "<p>Para executar os testes, adicione <code>?run_tests=1</code> à URL ou execute via linha de comando.</p>";
    echo "<p><strong>Importante:</strong> Configure sua API Key no arquivo antes de executar os testes.</p>";
    
    if (isset($_GET['run_tests'])) {
        echo "<pre>";
        runTests();
        echo "</pre>";
    }
}
?>
